/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.spark.k8s.operator.spec;

import java.time.Instant;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import io.fabric8.generator.annotation.Default;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import org.apache.spark.k8s.operator.status.ApplicationState;

/** Toleration settings for a Spark application. */
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
public class ApplicationTolerations {
  @Builder.Default protected RestartConfig restartConfig = new RestartConfig();

  @Builder.Default
  protected ApplicationTimeoutConfig applicationTimeoutConfig = new ApplicationTimeoutConfig();

  /** Determine the toleration behavior for executor / worker instances. */
  @Builder.Default protected ExecutorInstanceConfig instanceConfig = new ExecutorInstanceConfig();

  @Builder.Default protected ResourceRetainPolicy resourceRetainPolicy = ResourceRetainPolicy.Never;

  /**
   * Time-to-live in milliseconds for secondary resources of SparkApplications after termination. If
   * set to a negative value, secondary resources could be retained with the same lifecycle as the
   * application according to the retain policy.
   */
  @Default("-1")
  @Builder.Default
  protected Long resourceRetainDurationMillis = -1L;

  /**
   * Time-to-live in milliseconds for SparkApplication and all its associated secondary resources
   * after stop. If set to a negative value, the application could be retained according to the
   * retain policy. If the application is configured to restart, this would apply to the last
   * attempt only.
   */
  @Default("-1")
  @Builder.Default
  protected Long ttlAfterStopMillis = -1L;

  /**
   * Computes the effective resource retention duration in milliseconds. This is the smaller of
   * `resourceRetainDurationMillis` or `ttlAfterStopMillis` if both are non-negative. If only one is
   * non-negative, that value is used. If both are negative, -1L is returned.
   *
   * @return The effective retain duration for secondary resources, which would be the smaller value
   *     of `resourceRetainDurationMillis` or `ttlAfterStopMillis`, if they are set to non-negative
   *     value. Return -1 if none of them are set.
   */
  public long computeEffectiveRetainDurationMillis() {
    if (resourceRetainDurationMillis < 0 && ttlAfterStopMillis < 0) {
      return -1L;
    }
    if (resourceRetainDurationMillis < 0) {
      return ttlAfterStopMillis;
    }
    if (ttlAfterStopMillis < 0) {
      return resourceRetainDurationMillis;
    }
    return Math.min(resourceRetainDurationMillis, ttlAfterStopMillis);
  }

  /**
   * Check whether a terminated application has exceeded the resource retain duration at the
   * provided instant.
   *
   * @param lastObservedState The last observed state of the application.
   * @param instant The instant to check against.
   * @return True if the app has terminated, has a positive retain duration configured, and has
   *     exceeded that duration; false otherwise.
   */
  public boolean exceedRetainDurationAtInstant(
      ApplicationState lastObservedState, Instant instant) {
    return isRetainDurationEnabled()
        && lastObservedState != null
        && lastObservedState.getCurrentStateSummary().isTerminated()
        && Instant.parse(lastObservedState.getLastTransitionTime())
            .plusMillis(computeEffectiveRetainDurationMillis())
            .isBefore(instant);
  }

  /**
   * Indicates whether the reconciler needs to perform a retain duration check.
   *
   * @return True if `resourceRetainDurationMillis` or `ttlAfterStopMillis` is set to a non-negative
   *     value, false otherwise.
   */
  @JsonIgnore
  public boolean isRetainDurationEnabled() {
    return resourceRetainDurationMillis >= 0L || ttlAfterStopMillis >= 0L;
  }

  /**
   * Indicates whether the reconciler needs to perform a TTL (Time-To-Live) check.
   *
   * @return True if `ttlAfterStopMillis` is set to a non-negative value, false otherwise.
   */
  @JsonIgnore
  public boolean isTTLEnabled() {
    return ttlAfterStopMillis >= 0L;
  }
}
