/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.volume;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.fs.SpaceUsageCheckFactory;
import org.apache.hadoop.hdds.utils.HddsServerUtil;
import org.apache.hadoop.hdfs.server.datanode.StorageLocation;
import org.apache.hadoop.ozone.container.common.impl.StorageLocationReport;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.volume.DbVolumeFactory;
import org.apache.hadoop.ozone.container.common.volume.HddsVolume;
import org.apache.hadoop.ozone.container.common.volume.HddsVolumeFactory;
import org.apache.hadoop.ozone.container.common.volume.MetadataVolumeFactory;
import org.apache.hadoop.ozone.container.common.volume.StorageVolume;
import org.apache.hadoop.ozone.container.common.volume.StorageVolumeChecker;
import org.apache.hadoop.ozone.container.common.volume.StorageVolumeFactory;
import org.apache.hadoop.ozone.container.common.volume.VolumeHealthMetrics;
import org.apache.hadoop.ozone.container.common.volume.VolumeSet;
import org.apache.hadoop.util.DiskChecker;
import org.apache.ratis.util.function.CheckedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MutableVolumeSet
implements VolumeSet {
    private static final Logger LOG = LoggerFactory.getLogger(MutableVolumeSet.class);
    private ConfigurationSource conf;
    private Map<String, StorageVolume> volumeMap;
    private Map<String, StorageVolume> failedVolumeMap;
    private EnumMap<StorageType, List<StorageVolume>> volumeStateMap;
    private final ReentrantReadWriteLock volumeSetRWLock;
    private final String datanodeUuid;
    private final StorageVolumeChecker volumeChecker;
    private CheckedRunnable<IOException> failedVolumeListener;
    private StateContext context;
    private final StorageVolumeFactory volumeFactory;
    private final StorageVolume.VolumeType volumeType;
    private int maxVolumeFailuresTolerated;
    private final VolumeHealthMetrics volumeHealthMetrics;

    public MutableVolumeSet(String dnUuid, ConfigurationSource conf, StateContext context, StorageVolume.VolumeType volumeType, StorageVolumeChecker volumeChecker) throws IOException {
        this(dnUuid, null, conf, context, volumeType, volumeChecker);
    }

    public MutableVolumeSet(String dnUuid, String clusterID, ConfigurationSource conf, StateContext context, StorageVolume.VolumeType volumeType, StorageVolumeChecker volumeChecker) throws IOException {
        this.context = context;
        this.datanodeUuid = dnUuid;
        this.conf = conf;
        this.volumeSetRWLock = new ReentrantReadWriteLock();
        this.volumeChecker = volumeChecker;
        if (this.volumeChecker != null) {
            this.volumeChecker.registerVolumeSet(this);
        }
        this.volumeType = volumeType;
        SpaceUsageCheckFactory usageCheckFactory = SpaceUsageCheckFactory.create((ConfigurationSource)conf);
        DatanodeConfiguration dnConf = (DatanodeConfiguration)((Object)conf.getObject(DatanodeConfiguration.class));
        if (volumeType == StorageVolume.VolumeType.META_VOLUME) {
            this.volumeFactory = new MetadataVolumeFactory(conf, usageCheckFactory, this);
            this.maxVolumeFailuresTolerated = dnConf.getFailedMetadataVolumesTolerated();
        } else if (volumeType == StorageVolume.VolumeType.DB_VOLUME) {
            this.volumeFactory = new DbVolumeFactory(conf, usageCheckFactory, this, this.datanodeUuid, clusterID);
            this.maxVolumeFailuresTolerated = dnConf.getFailedDbVolumesTolerated();
        } else {
            this.volumeFactory = new HddsVolumeFactory(conf, usageCheckFactory, this, this.datanodeUuid, clusterID);
            this.maxVolumeFailuresTolerated = dnConf.getFailedDataVolumesTolerated();
        }
        this.volumeHealthMetrics = VolumeHealthMetrics.create(volumeType);
        try {
            this.initializeVolumeSet();
        }
        catch (Exception e) {
            this.volumeHealthMetrics.unregister();
            throw e;
        }
    }

    public void setFailedVolumeListener(CheckedRunnable<IOException> runnable) {
        this.failedVolumeListener = runnable;
    }

    @VisibleForTesting
    public StorageVolumeChecker getVolumeChecker() {
        return this.volumeChecker;
    }

    private void initializeVolumeSet() throws IOException {
        this.volumeMap = new ConcurrentHashMap<String, StorageVolume>();
        this.failedVolumeMap = new ConcurrentHashMap<String, StorageVolume>();
        this.volumeStateMap = new EnumMap(StorageType.class);
        Collection rawLocations = this.volumeType == StorageVolume.VolumeType.META_VOLUME ? HddsServerUtil.getOzoneDatanodeRatisDirectory((ConfigurationSource)this.conf) : (this.volumeType == StorageVolume.VolumeType.DB_VOLUME ? HddsServerUtil.getDatanodeDbDirs((ConfigurationSource)this.conf) : HddsServerUtil.getDatanodeStorageDirs((ConfigurationSource)this.conf));
        for (StorageType storageType : StorageType.values()) {
            this.volumeStateMap.put(storageType, new ArrayList());
        }
        for (String locationString : rawLocations) {
            StorageVolume volume = null;
            try {
                StorageLocation location = StorageLocation.parse((String)locationString);
                volume = this.volumeFactory.createVolume(location.getUri().getPath(), location.getStorageType());
                LOG.info("Added Volume : {} to VolumeSet", (Object)volume.getStorageDir().getPath());
                if (!volume.getStorageDir().mkdirs() && !volume.getStorageDir().exists()) {
                    throw new IOException("Failed to create storage dir " + volume.getStorageDir());
                }
                this.volumeMap.put(volume.getStorageDir().getPath(), volume);
                this.volumeStateMap.get(volume.getStorageType()).add(volume);
                this.volumeHealthMetrics.incrementHealthyVolumes();
            }
            catch (IOException e) {
                this.volumeHealthMetrics.incrementFailedVolumes();
                if (volume != null) {
                    volume.shutdown();
                }
                volume = this.volumeFactory.createFailedVolume(locationString);
                this.failedVolumeMap.put(locationString, volume);
                LOG.error("Failed to parse the storage location: " + locationString, (Throwable)e);
            }
        }
        if (this.volumeMap.isEmpty()) {
            throw new DiskChecker.DiskOutOfSpaceException("No storage locations configured");
        }
    }

    public void checkAllVolumes() throws IOException {
        this.checkAllVolumes(this.volumeChecker);
    }

    @Override
    public void checkAllVolumes(StorageVolumeChecker checker) throws IOException {
        Set<? extends StorageVolume> failedVolumes;
        if (checker == null) {
            LOG.debug("No volumeChecker, skip checkAllVolumes");
            return;
        }
        List<StorageVolume> allVolumes = this.getVolumesList();
        try {
            failedVolumes = checker.checkAllVolumes(allVolumes);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while running disk check", e);
        }
        if (!failedVolumes.isEmpty()) {
            LOG.warn("checkAllVolumes got {} failed volumes - {}", (Object)failedVolumes.size(), failedVolumes);
            this.handleVolumeFailures(failedVolumes);
        } else {
            LOG.debug("checkAllVolumes encountered no failures");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleVolumeFailures(Set<? extends StorageVolume> failedVolumes) throws IOException {
        this.writeLock();
        try {
            for (StorageVolume storageVolume : failedVolumes) {
                this.failVolume(storageVolume.getStorageDir().getPath());
            }
            if (!this.hasEnoughVolumes()) {
                this.context.getParent().handleFatalVolumeFailures();
            }
        }
        finally {
            this.writeUnlock();
        }
        if (this.failedVolumeListener != null) {
            this.failedVolumeListener.run();
        }
    }

    public void checkVolumeAsync(StorageVolume volume) {
        if (this.volumeChecker == null) {
            LOG.debug("No volumeChecker, skip checkVolumeAsync");
            return;
        }
        this.volumeChecker.checkVolume(volume, (healthyVolumes, failedVolumes) -> {
            if (!failedVolumes.isEmpty()) {
                LOG.warn("checkVolumeAsync callback got {} failed volumes: {}", (Object)failedVolumes.size(), (Object)failedVolumes);
            } else {
                LOG.debug("checkVolumeAsync: no volume failures detected");
            }
            this.handleVolumeFailures(failedVolumes);
        });
    }

    public void startAllVolume() throws IOException {
        for (Map.Entry<String, StorageVolume> entry : this.volumeMap.entrySet()) {
            entry.getValue().start();
        }
    }

    public void refreshAllVolumeUsage() {
        this.volumeMap.forEach((k, v) -> v.refreshVolumeUsage());
    }

    public void setGatherContainerUsages(Function<HddsVolume, Long> gatherContainerUsages) {
        this.volumeMap.forEach((k, v) -> v.setGatherContainerUsages(gatherContainerUsages));
    }

    public void readLock() {
        this.volumeSetRWLock.readLock().lock();
    }

    public void readUnlock() {
        this.volumeSetRWLock.readLock().unlock();
    }

    public void writeLock() {
        this.volumeSetRWLock.writeLock().lock();
    }

    public void writeUnlock() {
        this.volumeSetRWLock.writeLock().unlock();
    }

    boolean addVolume(String dataDir) {
        return this.addVolume(dataDir, StorageType.DEFAULT);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean addVolume(String volumeRoot, StorageType storageType) {
        boolean success;
        this.writeLock();
        try {
            if (this.volumeMap.containsKey(volumeRoot)) {
                LOG.warn("Volume : {} already exists in VolumeMap", (Object)volumeRoot);
                success = false;
            } else {
                if (this.failedVolumeMap.containsKey(volumeRoot)) {
                    this.failedVolumeMap.remove(volumeRoot);
                    this.volumeHealthMetrics.decrementFailedVolumes();
                }
                StorageVolume volume = this.volumeFactory.createVolume(volumeRoot, storageType);
                this.volumeMap.put(volume.getStorageDir().getPath(), volume);
                this.volumeStateMap.get(volume.getStorageType()).add(volume);
                LOG.info("Added Volume : {} to VolumeSet", (Object)volume.getStorageDir().getPath());
                success = true;
                this.volumeHealthMetrics.incrementHealthyVolumes();
            }
        }
        catch (IOException ex) {
            LOG.error("Failed to add volume " + volumeRoot + " to VolumeSet", (Throwable)ex);
            success = false;
        }
        finally {
            this.writeUnlock();
        }
        return success;
    }

    public void failVolume(String volumeRoot) {
        this.writeLock();
        try {
            if (this.volumeMap.containsKey(volumeRoot)) {
                StorageVolume volume = this.volumeMap.get(volumeRoot);
                volume.failVolume();
                this.volumeMap.remove(volumeRoot);
                this.volumeStateMap.get(volume.getStorageType()).remove(volume);
                this.failedVolumeMap.put(volumeRoot, volume);
                this.volumeHealthMetrics.decrementHealthyVolumes();
                this.volumeHealthMetrics.incrementFailedVolumes();
                LOG.info("Moving Volume : {} to failed Volumes", (Object)volumeRoot);
            } else if (this.failedVolumeMap.containsKey(volumeRoot)) {
                LOG.info("Volume : {} is not active", (Object)volumeRoot);
            } else {
                LOG.warn("Volume : {} does not exist in VolumeSet", (Object)volumeRoot);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    public void removeVolume(String volumeRoot) throws IOException {
        this.writeLock();
        try {
            if (this.volumeMap.containsKey(volumeRoot)) {
                StorageVolume volume = this.volumeMap.get(volumeRoot);
                volume.shutdown();
                this.volumeMap.remove(volumeRoot);
                this.volumeStateMap.get(volume.getStorageType()).remove(volume);
                this.volumeHealthMetrics.decrementHealthyVolumes();
                LOG.info("Removed Volume : {} from VolumeSet", (Object)volumeRoot);
            } else if (this.failedVolumeMap.containsKey(volumeRoot)) {
                this.failedVolumeMap.remove(volumeRoot);
                this.volumeHealthMetrics.decrementFailedVolumes();
                LOG.info("Removed Volume : {} from failed VolumeSet", (Object)volumeRoot);
            } else {
                LOG.warn("Volume : {} does not exist in VolumeSet", (Object)volumeRoot);
            }
        }
        finally {
            this.writeUnlock();
        }
    }

    public void shutdown() {
        for (StorageVolume volume : this.volumeMap.values()) {
            try {
                volume.shutdown();
            }
            catch (Exception ex) {
                LOG.error("Failed to shutdown volume : " + volume.getStorageDir(), (Throwable)ex);
            }
        }
        this.volumeMap.clear();
        if (this.volumeHealthMetrics != null) {
            this.volumeHealthMetrics.unregister();
        }
    }

    @Override
    @VisibleForTesting
    public List<StorageVolume> getVolumesList() {
        return ImmutableList.copyOf(this.volumeMap.values());
    }

    @VisibleForTesting
    public List<StorageVolume> getFailedVolumesList() {
        return ImmutableList.copyOf(this.failedVolumeMap.values());
    }

    @VisibleForTesting
    public Map<String, StorageVolume> getVolumeMap() {
        return ImmutableMap.copyOf(this.volumeMap);
    }

    @VisibleForTesting
    public void setVolumeMap(Map<String, StorageVolume> map) {
        this.volumeMap = map;
    }

    @VisibleForTesting
    public Map<StorageType, List<StorageVolume>> getVolumeStateMap() {
        return ImmutableMap.copyOf(this.volumeStateMap);
    }

    public boolean hasEnoughVolumes() {
        boolean hasEnoughVolumes;
        if (this.maxVolumeFailuresTolerated == -1) {
            hasEnoughVolumes = !this.getVolumesList().isEmpty();
        } else {
            boolean bl = hasEnoughVolumes = this.getFailedVolumesList().size() <= this.maxVolumeFailuresTolerated;
        }
        if (!hasEnoughVolumes) {
            LOG.error("Not enough volumes in MutableVolumeSet. DatanodeUUID: {}, VolumeType: {}, MaxVolumeFailuresTolerated: {}, ActiveVolumes: {}, FailedVolumes: {}", new Object[]{this.datanodeUuid, this.volumeType, this.maxVolumeFailuresTolerated, this.getVolumesList().size(), this.getFailedVolumesList().size()});
        }
        return hasEnoughVolumes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public StorageLocationReport[] getStorageReport() {
        this.readLock();
        try {
            StorageLocationReport[] reports = new StorageLocationReport[this.volumeMap.size() + this.failedVolumeMap.size()];
            int counter = 0;
            for (StorageVolume volume : this.volumeMap.values()) {
                reports[counter++] = volume.getReport();
            }
            for (StorageVolume volume : this.failedVolumeMap.values()) {
                reports[counter++] = volume.getReport();
            }
            StorageLocationReport[] storageLocationReportArray = reports;
            return storageLocationReportArray;
        }
        finally {
            this.readUnlock();
        }
    }

    public StorageVolume.VolumeType getVolumeType() {
        return this.volumeType;
    }

    @VisibleForTesting
    public VolumeHealthMetrics getVolumeHealthMetrics() {
        return this.volumeHealthMetrics;
    }
}

