/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.mailets;

import static org.apache.james.mailets.configuration.Constants.DEFAULT_DOMAIN;
import static org.apache.james.mailets.configuration.Constants.LOCALHOST_IP;
import static org.apache.james.mailets.configuration.Constants.PASSWORD;
import static org.apache.james.mailets.configuration.Constants.awaitAtMostOneMinute;
import static org.assertj.core.api.Assertions.assertThat;

import java.io.File;

import org.apache.james.MemoryJamesServerMain;
import org.apache.james.mailets.configuration.CommonProcessors;
import org.apache.james.mailets.configuration.MailetConfiguration;
import org.apache.james.mailets.configuration.ProcessorConfiguration;
import org.apache.james.mailrepository.api.MailRepositoryUrl;
import org.apache.james.modules.protocols.ImapGuiceProbe;
import org.apache.james.modules.protocols.SmtpGuiceProbe;
import org.apache.james.probe.DataProbe;
import org.apache.james.transport.mailets.ToProcessor;
import org.apache.james.transport.mailets.ToRepository;
import org.apache.james.transport.matchers.All;
import org.apache.james.transport.matchers.RemoteAddrInNetwork;
import org.apache.james.transport.matchers.RemoteAddrNotInNetwork;
import org.apache.james.utils.DataProbeImpl;
import org.apache.james.utils.MailRepositoryProbeImpl;
import org.apache.james.utils.SMTPMessageSender;
import org.apache.james.utils.TestIMAPClient;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.junit.jupiter.api.io.TempDir;

class NetworkMatcherIntegrationTest {
    private static final String FROM = "fromuser@" + DEFAULT_DOMAIN;
    private static final MailRepositoryUrl DROPPED_MAILS = MailRepositoryUrl.from("memory://var/mail/dropped-mails/");

    @RegisterExtension
    public TestIMAPClient testIMAPClient = new TestIMAPClient();
    @RegisterExtension
    public SMTPMessageSender messageSender = new SMTPMessageSender(DEFAULT_DOMAIN);

    private TemporaryJamesServer jamesServer;

    private TemporaryJamesServer createJamesServerWithRootProcessor(File temporaryFolder, ProcessorConfiguration.Builder rootProcessor) throws Exception {
        TemporaryJamesServer temporaryJamesServer = TemporaryJamesServer.builder()
            .withBase(MemoryJamesServerMain.SMTP_AND_IMAP_MODULE)
            .withMailetContainer(TemporaryJamesServer.defaultMailetContainerConfiguration()
                .putProcessor(rootProcessor)
                .putProcessor(CommonProcessors.deliverOnlyTransport()))
            .build(temporaryFolder);
        temporaryJamesServer.start();

        DataProbe dataProbe = temporaryJamesServer.getProbe(DataProbeImpl.class);
        dataProbe.addDomain(DEFAULT_DOMAIN);
        dataProbe.addUser(FROM, PASSWORD);
        return temporaryJamesServer;
    }

    private MailetConfiguration.Builder toRepository() {
        return MailetConfiguration.builder()
            .matcher(All.class)
            .mailet(ToRepository.class)
            .addProperty("repositoryPath", DROPPED_MAILS.asString());
    }

    @AfterEach
    void tearDown() {
        jamesServer.shutdown();
    }

    @Test
    void mailsFromAuthorizedNetworksShouldBeDeliveredWithRemoteAddrInNetwork(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrInNetwork.class)
                .matcherCondition("127.0.0.0/8")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
            .login(FROM, PASSWORD)
            .select(TestIMAPClient.INBOX)
            .awaitMessage(awaitAtMostOneMinute);
    }

    @Test
    void mailsFromAuthorizedNetworksShouldBeDeliveredWithRemoteAddrNotInNetwork(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrNotInNetwork.class)
                .matcherCondition("172.0.0.0/8")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
            .login(FROM, PASSWORD)
            .select(TestIMAPClient.INBOX)
            .awaitMessage(awaitAtMostOneMinute);
    }

    @Test
    void remoteAddrInNetworkShouldSupportLargerMask(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrInNetwork.class)
                .matcherCondition("127.0.0.0/2")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
            .login(FROM, PASSWORD)
            .select(TestIMAPClient.INBOX)
            .awaitMessage(awaitAtMostOneMinute);
    }

    @Test
    void remoteAddrInNetworkShouldSupportRangesDefinedByAMiddleIp(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrInNetwork.class)
                .matcherCondition("127.0.4.108/8")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
            .login(FROM, PASSWORD)
            .select(TestIMAPClient.INBOX)
            .awaitMessage(awaitAtMostOneMinute);
    }

    @Test
    void remoteAddrInNetworkShouldSupportRangesDefinedByEndingIp(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrInNetwork.class)
                .matcherCondition("127.255.255.255/8")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
            .login(FROM, PASSWORD)
            .select(TestIMAPClient.INBOX)
            .awaitMessage(awaitAtMostOneMinute);
    }

    @Test
    void remoteAddrInNetworkShouldSupportRangesWithNonEightMultipleSubMasks(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrInNetwork.class)
                .matcherCondition("126.0.0.0/4")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
            .login(FROM, PASSWORD)
            .select(TestIMAPClient.INBOX)
            .awaitMessage(awaitAtMostOneMinute);
    }

    @Test
    void mailsFromNonAuthorizedNetworksShouldNotBeDeliveredWithRemoteAddrInNetwork(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrInNetwork.class)
                .matcherCondition("172.0.0.0/8")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        MailRepositoryProbeImpl repositoryProbe = jamesServer.getProbe(MailRepositoryProbeImpl.class);
        awaitAtMostOneMinute.until(() -> repositoryProbe.getRepositoryMailCount(DROPPED_MAILS) == 1);
        assertThat(
            testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
                .login(FROM, PASSWORD)
                .select(TestIMAPClient.INBOX)
                .hasAMessage())
            .isFalse();
    }

    @Test
    void mailsFromNonAuthorizedNetworksShouldNotBeDeliveredWithRemoteAddrNotInNetwork(@TempDir File temporaryFolder) throws Exception {
        jamesServer = createJamesServerWithRootProcessor(temporaryFolder, ProcessorConfiguration.root()
            .addMailet(MailetConfiguration.builder()
                .matcher(RemoteAddrNotInNetwork.class)
                .matcherCondition("127.0.0.0/8")
                .mailet(ToProcessor.class)
                .addProperty("processor", ProcessorConfiguration.TRANSPORT_PROCESSOR))
            .addMailet(toRepository()));

        messageSender.connect(LOCALHOST_IP, jamesServer.getProbe(SmtpGuiceProbe.class).getSmtpPort())
            .authenticate(FROM, PASSWORD)
            .sendMessage(FROM, FROM);

        MailRepositoryProbeImpl repositoryProbe = jamesServer.getProbe(MailRepositoryProbeImpl.class);
        awaitAtMostOneMinute.until(() -> repositoryProbe.getRepositoryMailCount(DROPPED_MAILS) == 1);
        assertThat(
            testIMAPClient.connect(LOCALHOST_IP, jamesServer.getProbe(ImapGuiceProbe.class).getImapPort())
                .login(FROM, PASSWORD)
                .select(TestIMAPClient.INBOX)
                .hasAMessage())
            .isFalse();
    }

}
