/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import java.io.IOException;
import java.util.HashSet;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Random;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.raft.CandidateState;
import org.apache.kafka.raft.ElectionState;
import org.apache.kafka.raft.EpochState;
import org.apache.kafka.raft.FollowerState;
import org.apache.kafka.raft.LeaderAndEpoch;
import org.apache.kafka.raft.LeaderState;
import org.apache.kafka.raft.LogOffsetMetadata;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.raft.QuorumStateStore;
import org.apache.kafka.raft.UnattachedState;
import org.apache.kafka.raft.VotedState;
import org.slf4j.Logger;

public class QuorumState {
    public final int localId;
    private final Time time;
    private final Logger log;
    private final QuorumStateStore store;
    private final Set<Integer> voters;
    private final Random random;
    private final int electionTimeoutMs;
    private final int fetchTimeoutMs;
    private volatile EpochState state;

    public QuorumState(int localId, Set<Integer> voters, int electionTimeoutMs, int fetchTimeoutMs, QuorumStateStore store, Time time, LogContext logContext, Random random) {
        this.localId = localId;
        this.voters = new HashSet<Integer>(voters);
        this.electionTimeoutMs = electionTimeoutMs;
        this.fetchTimeoutMs = fetchTimeoutMs;
        this.store = store;
        this.time = time;
        this.log = logContext.logger(QuorumState.class);
        this.random = random;
    }

    public void initialize(OffsetAndEpoch logEndOffsetAndEpoch) throws IOException, IllegalStateException {
        EpochState initialState;
        ElectionState election;
        try {
            election = this.store.readElectionState();
            if (election == null) {
                election = ElectionState.withUnknownLeader(0, this.voters);
            }
        }
        catch (IOException e) {
            this.log.warn("Clearing local quorum state store after error loading state {}", (Object)this.store.toString(), (Object)e);
            this.store.clear();
            election = ElectionState.withUnknownLeader(0, this.voters);
        }
        if (!election.voters().isEmpty() && !this.voters.equals(election.voters())) {
            throw new IllegalStateException("Configured voter set: " + this.voters + " is different from the voter set read from the state file: " + election.voters() + ". Check if the quorum configuration is up to date, or wipe out the local state file if necessary");
        }
        if (election.epoch < logEndOffsetAndEpoch.epoch) {
            this.log.warn("Epoch from quorum-state file is {}, which is smaller than last written epoch {} in the log", (Object)election.epoch, (Object)logEndOffsetAndEpoch.epoch);
            initialState = new UnattachedState(this.time, logEndOffsetAndEpoch.epoch, this.voters, this.randomElectionTimeoutMs());
        } else {
            initialState = election.isLeader(this.localId) ? new UnattachedState(this.time, election.epoch, this.voters, this.randomElectionTimeoutMs()) : (election.isCandidate(this.localId) ? new CandidateState(this.time, this.localId, election.epoch, this.voters, 1, this.randomElectionTimeoutMs()) : (election.hasVoted() ? new VotedState(this.time, election.epoch, election.votedId(), this.voters, this.randomElectionTimeoutMs()) : (election.hasLeader() ? new FollowerState(this.time, election.epoch, election.leaderId(), this.voters, this.fetchTimeoutMs) : new UnattachedState(this.time, election.epoch, this.voters, this.randomElectionTimeoutMs()))));
        }
        this.transitionTo(initialState);
    }

    public Set<Integer> remoteVoters() {
        return this.voters.stream().filter(voterId -> voterId != this.localId).collect(Collectors.toSet());
    }

    public int epoch() {
        return this.state.epoch();
    }

    public int leaderIdOrNil() {
        return this.leaderId().orElse(-1);
    }

    public Optional<LogOffsetMetadata> highWatermark() {
        return this.state.highWatermark();
    }

    public OptionalInt leaderId() {
        ElectionState election = this.state.election();
        if (election.hasLeader()) {
            return OptionalInt.of(this.state.election().leaderId());
        }
        return OptionalInt.empty();
    }

    public boolean hasLeader() {
        return this.leaderId().isPresent();
    }

    public boolean hasRemoteLeader() {
        return this.hasLeader() && this.leaderIdOrNil() != this.localId;
    }

    public boolean isVoter() {
        return this.voters.contains(this.localId);
    }

    public boolean isVoter(int nodeId) {
        return this.voters.contains(nodeId);
    }

    public boolean isObserver() {
        return !this.isVoter();
    }

    public void transitionToUnattached(int epoch) throws IOException {
        int currentEpoch = this.state.epoch();
        if (epoch <= currentEpoch) {
            throw new IllegalStateException("Cannot transition to Unattached with epoch= " + epoch + " from current state " + this.state);
        }
        long electionTimeoutMs = this.isObserver() ? Long.MAX_VALUE : (this.isCandidate() ? this.candidateStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (this.isVoted() ? this.votedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (this.isUnattached() ? this.unattachedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (long)this.randomElectionTimeoutMs())));
        this.transitionTo(new UnattachedState(this.time, epoch, this.voters, electionTimeoutMs));
    }

    public void transitionToVoted(int epoch, int candidateId) throws IOException {
        if (candidateId == this.localId) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " since it matches the local broker.id");
        }
        if (this.isObserver()) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " since the local broker.id=" + this.localId + " is not a voter");
        }
        if (!this.isVoter(candidateId)) {
            throw new IllegalStateException("Cannot transition to Voted with voterId=" + candidateId + " and epoch=" + epoch + " since it is not one of the voters " + this.voters);
        }
        int currentEpoch = this.state.epoch();
        if (epoch < currentEpoch) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " since the current epoch " + currentEpoch + " is larger");
        }
        if (epoch == currentEpoch && !this.isUnattached()) {
            throw new IllegalStateException("Cannot transition to Voted with votedId=" + candidateId + " and epoch=" + epoch + " from the current state " + this.state);
        }
        this.transitionTo(new VotedState(this.time, epoch, candidateId, this.voters, this.randomElectionTimeoutMs()));
    }

    public void transitionToFollower(int epoch, int leaderId) throws IOException {
        if (leaderId == this.localId) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " since it matches the local broker.id=" + this.localId);
        }
        if (!this.isVoter(leaderId)) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " since it is not one of the voters " + this.voters);
        }
        int currentEpoch = this.state.epoch();
        if (epoch < currentEpoch) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " since the current epoch " + currentEpoch + " is larger");
        }
        if (epoch == currentEpoch && (this.isFollower() || this.isLeader())) {
            throw new IllegalStateException("Cannot transition to Follower with leaderId=" + leaderId + " and epoch=" + epoch + " from state " + this.state);
        }
        this.transitionTo(new FollowerState(this.time, epoch, leaderId, this.voters, this.fetchTimeoutMs));
    }

    public void transitionToCandidate() throws IOException {
        if (this.isObserver()) {
            throw new IllegalStateException("Cannot transition to Candidate since the local broker.id=" + this.localId + " is not one of the voters " + this.voters);
        }
        if (this.isLeader()) {
            throw new IllegalStateException("Cannot transition to Candidate since the local broker.id=" + this.localId + " since this node is already a Leader with state " + this.state);
        }
        int retries = this.isCandidate() ? this.candidateStateOrThrow().retries() + 1 : 1;
        int newEpoch = this.epoch() + 1;
        int electionTimeoutMs = this.randomElectionTimeoutMs();
        this.transitionTo(new CandidateState(this.time, this.localId, newEpoch, this.voters, retries, electionTimeoutMs));
    }

    public void transitionToLeader(long epochStartOffset) throws IOException {
        if (this.isObserver()) {
            throw new IllegalStateException("Cannot transition to Leader since the local broker.id=" + this.localId + " is not one of the voters " + this.voters);
        }
        if (!this.isCandidate()) {
            throw new IllegalStateException("Cannot transition to Leader from current state " + this.state);
        }
        CandidateState candidateState = this.candidateStateOrThrow();
        if (!candidateState.isVoteGranted()) {
            throw new IllegalStateException("Cannot become leader without majority votes granted");
        }
        this.transitionTo(new LeaderState(this.localId, this.epoch(), epochStartOffset, this.voters));
    }

    private void transitionTo(EpochState state) throws IOException {
        this.store.writeElectionState(state.election());
        this.state = state;
        this.log.info("Completed transition to {}", (Object)state);
    }

    private int randomElectionTimeoutMs() {
        if (this.electionTimeoutMs == 0) {
            return 0;
        }
        return this.electionTimeoutMs + this.random.nextInt(this.electionTimeoutMs);
    }

    public FollowerState followerStateOrThrow() {
        if (this.isFollower()) {
            return (FollowerState)this.state;
        }
        throw new IllegalStateException("Expected to be Follower, but the current state is " + this.state);
    }

    public VotedState votedStateOrThrow() {
        if (this.isVoted()) {
            return (VotedState)this.state;
        }
        throw new IllegalStateException("Expected to be Voted, but current state is " + this.state);
    }

    public UnattachedState unattachedStateOrThrow() {
        if (this.isUnattached()) {
            return (UnattachedState)this.state;
        }
        throw new IllegalStateException("Expected to be Unattached, but current state is " + this.state);
    }

    public LeaderState leaderStateOrThrow() {
        if (this.isLeader()) {
            return (LeaderState)this.state;
        }
        throw new IllegalStateException("Expected to be Leader, but current state is " + this.state);
    }

    public CandidateState candidateStateOrThrow() {
        if (this.isCandidate()) {
            return (CandidateState)this.state;
        }
        throw new IllegalStateException("Expected to be Candidate, but current state is " + this.state);
    }

    public LeaderAndEpoch leaderAndEpoch() {
        ElectionState election = this.state.election();
        return new LeaderAndEpoch(election.leaderIdOpt, election.epoch);
    }

    public boolean isFollower() {
        return this.state instanceof FollowerState;
    }

    public boolean isVoted() {
        return this.state instanceof VotedState;
    }

    public boolean isUnattached() {
        return this.state instanceof UnattachedState;
    }

    public boolean isLeader() {
        return this.state instanceof LeaderState;
    }

    public boolean isCandidate() {
        return this.state instanceof CandidateState;
    }
}

