/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.metastore;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileChecksum;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.metastore.api.Database;
import org.apache.hadoop.hive.metastore.api.MetaException;
import org.apache.hadoop.hive.metastore.conf.MetastoreConf;
import org.apache.hadoop.hive.metastore.utils.FileUtils;
import org.apache.hadoop.hive.metastore.utils.StringUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplChangeManager {
    private static final Logger LOG = LoggerFactory.getLogger(ReplChangeManager.class);
    private static ReplChangeManager instance;
    private static boolean inited;
    private static boolean enabled;
    private static Path cmroot;
    private static Configuration conf;
    private String msUser;
    private String msGroup;
    private static final String ORIG_LOC_TAG = "user.original-loc";
    static final String REMAIN_IN_TRASH_TAG = "user.remain-in-trash";
    private static final String URI_FRAGMENT_SEPARATOR = "#";
    public static final String SOURCE_OF_REPLICATION = "repl.source.for";
    private static final PathFilter hiddenFileFilter;

    public static ReplChangeManager getInstance(Configuration conf) throws MetaException {
        if (instance == null) {
            instance = new ReplChangeManager(conf);
        }
        return instance;
    }

    private ReplChangeManager(Configuration conf) throws MetaException {
        try {
            if (!inited) {
                if (MetastoreConf.getBoolVar(conf, MetastoreConf.ConfVars.REPLCMENABLED)) {
                    enabled = true;
                    cmroot = new Path(MetastoreConf.getVar(conf, MetastoreConf.ConfVars.REPLCMDIR));
                    ReplChangeManager.conf = conf;
                    FileSystem cmFs = cmroot.getFileSystem(conf);
                    if (!cmFs.exists(cmroot)) {
                        cmFs.mkdirs(cmroot);
                        cmFs.setPermission(cmroot, new FsPermission("700"));
                    }
                    UserGroupInformation usergroupInfo = UserGroupInformation.getCurrentUser();
                    this.msUser = usergroupInfo.getShortUserName();
                    this.msGroup = usergroupInfo.getPrimaryGroupName();
                }
                inited = true;
            }
        }
        catch (IOException e) {
            throw new MetaException(StringUtils.stringifyException(e));
        }
    }

    public int recycle(Path path, RecycleType type, boolean ifPurge) throws IOException {
        int count;
        block16: {
            FileSystem fs;
            block15: {
                FileStatus[] files;
                if (!enabled) {
                    return 0;
                }
                count = 0;
                fs = path.getFileSystem(conf);
                if (!fs.isDirectory(path)) break block15;
                for (FileStatus file : files = fs.listStatus(path, hiddenFileFilter)) {
                    count += this.recycle(file.getPath(), type, ifPurge);
                }
                break block16;
            }
            String fileCheckSum = ReplChangeManager.checksumFor(path, fs);
            Path cmPath = ReplChangeManager.getCMPath(conf, path.getName(), fileCheckSum, cmroot.toString());
            long now = System.currentTimeMillis();
            fs.setTimes(path, now, -1L);
            boolean success = false;
            if (fs.exists(cmPath) && fileCheckSum.equalsIgnoreCase(ReplChangeManager.checksumFor(cmPath, fs))) {
                success = false;
            } else {
                switch (type) {
                    case MOVE: {
                        LOG.info("Moving {} to {}", (Object)path.toString(), (Object)cmPath.toString());
                        success = fs.rename(path, cmPath);
                        break;
                    }
                    case COPY: {
                        LOG.info("Copying {} to {}", (Object)path.toString(), (Object)cmPath.toString());
                        success = FileUtils.copy(fs, path, fs, cmPath, false, true, conf);
                        break;
                    }
                }
            }
            if (success) {
                fs.setOwner(cmPath, this.msUser, this.msGroup);
                try {
                    fs.setXAttr(cmPath, ORIG_LOC_TAG, path.toString().getBytes());
                }
                catch (UnsupportedOperationException e) {
                    LOG.warn("Error setting xattr for {}", (Object)path.toString());
                }
                ++count;
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("A file with the same content of {} already exists, ignore", (Object)path.toString());
                }
                fs.setTimes(cmPath, now, -1L);
            }
            if (type != RecycleType.MOVE || ifPurge) break block16;
            try {
                fs.setXAttr(cmPath, REMAIN_IN_TRASH_TAG, new byte[]{0});
            }
            catch (UnsupportedOperationException e) {
                LOG.warn("Error setting xattr for {}", (Object)cmPath.toString());
            }
        }
        return count;
    }

    public static String checksumFor(Path path, FileSystem fs) throws IOException {
        String checksumString = null;
        FileChecksum checksum = fs.getFileChecksum(path);
        if (checksum != null) {
            checksumString = StringUtils.byteToHexString(checksum.getBytes(), 0, checksum.getLength());
        }
        return checksumString;
    }

    static Path getCMPath(Configuration conf, String name, String checkSum, String cmRootUri) {
        String newFileName = name + "_" + checkSum;
        int maxLength = conf.getInt("dfs.namenode.fs-limits.max-component-length", 255);
        if (newFileName.length() > maxLength) {
            newFileName = newFileName.substring(0, maxLength - 1);
        }
        return new Path(cmRootUri, newFileName);
    }

    public static FileInfo getFileInfo(Path src, String checksumString, String srcCMRootURI, String subDir, Configuration conf) throws MetaException {
        try {
            String currentChecksumString;
            FileSystem srcFs = src.getFileSystem(conf);
            if (checksumString == null) {
                return new FileInfo(srcFs, src, subDir);
            }
            Path cmPath = ReplChangeManager.getCMPath(conf, src.getName(), checksumString, srcCMRootURI);
            if (!srcFs.exists(src)) {
                return new FileInfo(srcFs, src, cmPath, checksumString, false, subDir);
            }
            try {
                currentChecksumString = ReplChangeManager.checksumFor(src, srcFs);
            }
            catch (IOException ex) {
                return new FileInfo(srcFs, src, cmPath, checksumString, false, subDir);
            }
            if (currentChecksumString == null || checksumString.equals(currentChecksumString)) {
                return new FileInfo(srcFs, src, cmPath, checksumString, true, subDir);
            }
            return new FileInfo(srcFs, src, cmPath, checksumString, false, subDir);
        }
        catch (IOException e) {
            throw new MetaException(StringUtils.stringifyException(e));
        }
    }

    public static String encodeFileUri(String fileUriStr, String fileChecksum, String encodedSubDir) throws IOException {
        String encodedUri = fileUriStr;
        encodedUri = fileChecksum != null && cmroot != null ? encodedUri + URI_FRAGMENT_SEPARATOR + fileChecksum + URI_FRAGMENT_SEPARATOR + FileUtils.makeQualified(cmroot, conf) : encodedUri + URI_FRAGMENT_SEPARATOR + URI_FRAGMENT_SEPARATOR;
        encodedUri = encodedUri + URI_FRAGMENT_SEPARATOR + (encodedSubDir != null ? encodedSubDir : "");
        if (LOG.isDebugEnabled()) {
            LOG.debug("Encoded URI: " + encodedUri);
        }
        return encodedUri;
    }

    public static String[] decodeFileUri(String fileURIStr) {
        String[] uriAndFragment = fileURIStr.split(URI_FRAGMENT_SEPARATOR);
        String[] result = new String[4];
        result[0] = uriAndFragment[0];
        if (uriAndFragment.length > 1 && !StringUtils.isEmpty(uriAndFragment[1])) {
            result[1] = uriAndFragment[1];
        }
        if (uriAndFragment.length > 2 && !StringUtils.isEmpty(uriAndFragment[2])) {
            result[2] = uriAndFragment[2];
        }
        if (uriAndFragment.length > 3 && !StringUtils.isEmpty(uriAndFragment[3])) {
            result[3] = uriAndFragment[3];
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Reading Encoded URI: " + result[0] + ":: " + result[1] + ":: " + result[2] + ":: " + result[3]);
        }
        return result;
    }

    public static boolean isCMFileUri(Path fromPath) {
        String[] result = ReplChangeManager.decodeFileUri(fromPath.toString());
        return result[1] != null;
    }

    static void scheduleCMClearer(Configuration conf) {
        if (MetastoreConf.getBoolVar(conf, MetastoreConf.ConfVars.REPLCMENABLED)) {
            ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor((ThreadFactory)new BasicThreadFactory.Builder().namingPattern("cmclearer-%d").daemon(true).build());
            executor.scheduleAtFixedRate(new CMClearer(MetastoreConf.getVar(conf, MetastoreConf.ConfVars.REPLCMDIR), MetastoreConf.getTimeVar(conf, MetastoreConf.ConfVars.REPLCMRETIAN, TimeUnit.SECONDS), conf), 0L, MetastoreConf.getTimeVar(conf, MetastoreConf.ConfVars.REPLCMINTERVAL, TimeUnit.SECONDS), TimeUnit.SECONDS);
        }
    }

    public static boolean isSourceOfReplication(Database db) {
        assert (db != null);
        String replPolicyIds = ReplChangeManager.getReplPolicyIdString(db);
        return !StringUtils.isEmpty(replPolicyIds);
    }

    public static String getReplPolicyIdString(Database db) {
        if (db != null) {
            Map<String, String> m = db.getParameters();
            if (m != null && m.containsKey(SOURCE_OF_REPLICATION)) {
                String replPolicyId = m.get(SOURCE_OF_REPLICATION);
                LOG.debug("repl policy for database {} is {}", (Object)db.getName(), (Object)replPolicyId);
                return replPolicyId;
            }
            LOG.debug("Repl policy is not set for database ", (Object)db.getName());
        }
        return null;
    }

    static {
        inited = false;
        enabled = false;
        hiddenFileFilter = new PathFilter(){

            public boolean accept(Path p) {
                return !p.getName().startsWith(".");
            }
        };
    }

    static class CMClearer
    implements Runnable {
        private Path cmroot;
        private long secRetain;
        private Configuration conf;

        CMClearer(String cmrootString, long secRetain, Configuration conf) {
            this.cmroot = new Path(cmrootString);
            this.secRetain = secRetain;
            this.conf = conf;
        }

        @Override
        public void run() {
            try {
                FileStatus[] files;
                LOG.info("CMClearer started");
                long now = System.currentTimeMillis();
                FileSystem fs = this.cmroot.getFileSystem(this.conf);
                for (FileStatus file : files = fs.listStatus(this.cmroot)) {
                    long modifiedTime = file.getModificationTime();
                    if (now - modifiedTime <= this.secRetain * 1000L) continue;
                    try {
                        boolean succ;
                        if (fs.getXAttrs(file.getPath()).containsKey(ReplChangeManager.REMAIN_IN_TRASH_TAG)) {
                            succ = Trash.moveToAppropriateTrash((FileSystem)fs, (Path)file.getPath(), (Configuration)this.conf);
                            if (succ) {
                                if (!LOG.isDebugEnabled()) continue;
                                LOG.debug("Move " + file.toString() + " to trash");
                                continue;
                            }
                            LOG.warn("Fail to move " + file.toString() + " to trash");
                            continue;
                        }
                        succ = fs.delete(file.getPath(), false);
                        if (succ) {
                            if (!LOG.isDebugEnabled()) continue;
                            LOG.debug("Remove " + file.toString());
                            continue;
                        }
                        LOG.warn("Fail to remove " + file.toString());
                    }
                    catch (UnsupportedOperationException e) {
                        LOG.warn("Error getting xattr for " + file.getPath().toString());
                    }
                }
            }
            catch (IOException e) {
                LOG.error("Exception when clearing cmroot:" + StringUtils.stringifyException(e));
            }
        }
    }

    public static class FileInfo {
        private FileSystem srcFs;
        private Path sourcePath;
        private Path cmPath;
        private String checkSum;
        private boolean useSourcePath;
        private String subDir;
        private boolean copyDone;

        public FileInfo(FileSystem srcFs, Path sourcePath, String subDir) {
            this(srcFs, sourcePath, null, null, true, subDir);
        }

        public FileInfo(FileSystem srcFs, Path sourcePath, Path cmPath, String checkSum, boolean useSourcePath, String subDir) {
            this.srcFs = srcFs;
            this.sourcePath = sourcePath;
            this.cmPath = cmPath;
            this.checkSum = checkSum;
            this.useSourcePath = useSourcePath;
            this.subDir = subDir;
            this.copyDone = false;
        }

        public FileSystem getSrcFs() {
            return this.srcFs;
        }

        public Path getSourcePath() {
            return this.sourcePath;
        }

        public Path getCmPath() {
            return this.cmPath;
        }

        public String getCheckSum() {
            return this.checkSum;
        }

        public boolean isUseSourcePath() {
            return this.useSourcePath;
        }

        public void setIsUseSourcePath(boolean useSourcePath) {
            this.useSourcePath = useSourcePath;
        }

        public String getSubDir() {
            return this.subDir;
        }

        public boolean isCopyDone() {
            return this.copyDone;
        }

        public void setCopyDone(boolean copyDone) {
            this.copyDone = copyDone;
        }

        public Path getEffectivePath() {
            if (this.useSourcePath) {
                return this.sourcePath;
            }
            return this.cmPath;
        }
    }

    public static enum RecycleType {
        MOVE,
        COPY;

    }
}

