/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hop.pipeline.transforms.calculator;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.fail;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.spy;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import org.apache.hop.core.Const;
import org.apache.hop.core.HopEnvironment;
import org.apache.hop.core.IRowSet;
import org.apache.hop.core.exception.HopException;
import org.apache.hop.core.exception.HopTransformException;
import org.apache.hop.core.logging.ILoggingObject;
import org.apache.hop.core.row.IRowMeta;
import org.apache.hop.core.row.IValueMeta;
import org.apache.hop.core.row.RowMeta;
import org.apache.hop.core.row.value.ValueMetaBigNumber;
import org.apache.hop.core.row.value.ValueMetaDate;
import org.apache.hop.core.row.value.ValueMetaFactory;
import org.apache.hop.core.row.value.ValueMetaInteger;
import org.apache.hop.core.row.value.ValueMetaNumber;
import org.apache.hop.core.row.value.ValueMetaString;
import org.apache.hop.junit.rules.RestoreHopEngineEnvironmentExtension;
import org.apache.hop.pipeline.transform.RowAdapter;
import org.apache.hop.pipeline.transforms.mock.TransformMockHelper;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

/**
 * Unit tests for calculator transform
 *
 * @see Calculator
 */
class CalculatorUnitTest {
  private TransformMockHelper<CalculatorMeta, CalculatorData> smh;

  @RegisterExtension
  static RestoreHopEngineEnvironmentExtension env = new RestoreHopEngineEnvironmentExtension();

  @BeforeAll
  static void init() throws HopException {
    HopEnvironment.init();
  }

  @BeforeEach
  void setUp() {
    smh = new TransformMockHelper<>("Calculator", CalculatorMeta.class, CalculatorData.class);
    when(smh.logChannelFactory.create(any(), any(ILoggingObject.class)))
        .thenReturn(smh.iLogChannel);
    when(smh.pipeline.isRunning()).thenReturn(true);
  }

  @AfterEach
  void cleanUp() {
    smh.cleanUp();
  }

  @Test
  void testMissingFile() throws HopException {
    RowMeta inputRowMeta = new RowMeta();
    ValueMetaString pathMeta = new ValueMetaString("Path");
    inputRowMeta.addValueMeta(pathMeta);

    String filepath = "missingFile";
    Object[] rows = new Object[] {filepath};
    IRowSet inputRowSet = smh.getMockInputRowSet(rows);
    inputRowSet.setRowMeta(inputRowMeta);

    CalculatorMeta meta = new CalculatorMeta();
    meta.getFunctions()
        .add(
            new CalculatorMetaFunction(
                "result",
                CalculationType.MD5,
                "Path",
                null,
                null,
                "String",
                0,
                0,
                "",
                "",
                "",
                "",
                false));
    meta.setFailIfNoFile(true);

    CalculatorData data = new CalculatorData();

    Calculator calculator =
        spy(new Calculator(smh.transformMeta, meta, data, 0, smh.pipelineMeta, smh.pipeline));
    calculator.addRowSetToInputRowSets(inputRowSet);
    calculator.setInputRowMeta(inputRowMeta);
    calculator.init();

    boolean processed = calculator.processRow();
    assertFalse(processed);
  }

  @Test
  void testAddSeconds() {
    RowMeta inputRowMeta = new RowMeta();
    ValueMetaDate dayMeta = new ValueMetaDate("Day");
    inputRowMeta.addValueMeta(dayMeta);
    ValueMetaInteger secondsMeta = new ValueMetaInteger("Seconds");
    inputRowMeta.addValueMeta(secondsMeta);

    IRowSet inputRowSet = null;
    try {
      inputRowSet =
          smh.getMockInputRowSet(
              new Object[][] {
                {
                  new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse("2014-01-01 00:00:00"),
                  Long.valueOf(10)
                },
                {
                  new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse("2014-10-31 23:59:50"),
                  Long.valueOf(30)
                }
              });
    } catch (ParseException pe) {
      pe.printStackTrace();
      fail();
    }
    inputRowSet.setRowMeta(inputRowMeta);

    CalculatorMeta meta = new CalculatorMeta();
    meta.getFunctions()
        .add(
            new CalculatorMetaFunction(
                "new_day",
                CalculationType.ADD_SECONDS,
                "Day",
                "Seconds",
                null,
                "Date",
                0,
                0,
                "",
                "",
                "",
                "",
                false));

    CalculatorData data = new CalculatorData();

    Calculator calculator =
        new Calculator(smh.transformMeta, meta, data, 0, smh.pipelineMeta, smh.pipeline);
    calculator.addRowSetToInputRowSets(inputRowSet);
    calculator.setInputRowMeta(inputRowMeta);
    calculator.init();

    // Verify output
    try {
      calculator.addRowListener(
          new RowAdapter() {
            @Override
            public void rowWrittenEvent(IRowMeta rowMeta, Object[] row)
                throws HopTransformException {
              try {
                customAssertEquals(
                    new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").parse("2014-01-01 00:00:10"),
                    row[2]);
              } catch (ParseException pe) {
                throw new HopTransformException(pe);
              }
            }
          });
      calculator.processRow();
    } catch (HopException ke) {
      ke.printStackTrace();
      fail();
    }
  }

  @Test
  void testReturnDigitsOnly() {
    RowMeta inputRowMeta = new RowMeta();
    ValueMetaString nameMeta = new ValueMetaString("Name");
    inputRowMeta.addValueMeta(nameMeta);
    ValueMetaString valueMeta = new ValueMetaString("Value");
    inputRowMeta.addValueMeta(valueMeta);

    IRowSet inputRowSet =
        smh.getMockInputRowSet(new Object[][] {{"name1", "qwe123asd456zxc"}, {"name2", null}});
    inputRowSet.setRowMeta(inputRowMeta);

    CalculatorMeta meta = new CalculatorMeta();
    meta.getFunctions()
        .add(
            new CalculatorMetaFunction(
                "digits",
                CalculationType.GET_ONLY_DIGITS,
                "Value",
                null,
                null,
                "String",
                0,
                0,
                "",
                "",
                "",
                "",
                false));

    CalculatorData data = new CalculatorData();

    Calculator calculator =
        new Calculator(smh.transformMeta, meta, data, 0, smh.pipelineMeta, smh.pipeline);
    calculator.addRowSetToInputRowSets(inputRowSet);
    calculator.setInputRowMeta(inputRowMeta);
    calculator.init();

    // Verify output
    try {
      calculator.addRowListener(
          new RowAdapter() {
            @Override
            public void rowWrittenEvent(IRowMeta rowMeta, Object[] row) {
              customAssertEquals("123456", row[2]);
            }
          });
      calculator.processRow();
    } catch (HopException ke) {
      ke.printStackTrace();
      fail();
    }
  }

  @Test
  void calculatorShouldClearDataInstance() throws Exception {
    RowMeta inputRowMeta = new RowMeta();
    ValueMetaInteger valueMeta = new ValueMetaInteger("Value");
    inputRowMeta.addValueMeta(valueMeta);

    IRowSet inputRowSet = smh.getMockInputRowSet(new Object[] {-1L});
    inputRowSet.setRowMeta(inputRowMeta);

    CalculatorMeta meta = new CalculatorMeta();
    meta.getFunctions()
        .add(
            new CalculatorMetaFunction(
                "test",
                CalculationType.ABS,
                "Value",
                null,
                null,
                "String",
                0,
                0,
                "",
                "",
                "",
                "",
                false));

    CalculatorData data = spy(new CalculatorData());

    Calculator calculator =
        new Calculator(smh.transformMeta, meta, data, 0, smh.pipelineMeta, smh.pipeline);
    calculator.addRowSetToInputRowSets(inputRowSet);
    calculator.setInputRowMeta(inputRowMeta);
    calculator.init();

    calculator.processRow();
    verify(data).getValueMetaFor(eq(valueMeta.getType()), anyString());

    calculator.processRow();
    verify(data).clearValuesMetaMapping();
  }

  @Test
  void testRound1() throws HopException {
    assertRound1(1.0, 1.2);
    assertRound1(2.0, 1.5);
    assertRound1(2.0, 1.7);
    assertRound1(2.0, 2.2);
    assertRound1(3.0, 2.5);
    assertRound1(3.0, 2.7);
    assertRound1(-1.0, -1.2);
    assertRound1(-1.0, -1.5);
    assertRound1(-2.0, -1.7);
    assertRound1(-2.0, -2.2);
    assertRound1(-2.0, -2.5);
    assertRound1(-3.0, -2.7);
    assertRound1(1.0, 1.0);
    assertRound1(2.0, 2.0);
    assertRound1(-3.0, -3.0);
  }

  @Test
  void testRound2() throws HopException {
    assertRound2(1.0, 1.2, 0);
    assertRound2(2.0, 1.5, 0);
    assertRound2(2.0, 1.7, 0);
    assertRound2(2.0, 2.2, 0);
    assertRound2(3.0, 2.5, 0);
    assertRound2(3.0, 2.7, 0);
    assertRound2(-1.0, -1.2, 0);
    assertRound2(-1.0, -1.5, 0);
    assertRound2(-2.0, -1.7, 0);
    assertRound2(-2.0, -2.2, 0);
    assertRound2(-2.0, -2.5, 0);
    assertRound2(-3.0, -2.7, 0);
    assertRound2(1.0, 1.0, 0);
    assertRound2(2.0, 2.0, 0);
    assertRound2(-3.0, -3.0, 0);

    assertRound2(0.010, 0.012, 2);
    assertRound2(0.020, 0.015, 2);
    assertRound2(0.020, 0.017, 2);
    assertRound2(0.020, 0.022, 2);
    assertRound2(0.030, 0.025, 2);
    assertRound2(0.030, 0.027, 2);
    assertRound2(-0.010, -0.012, 2);
    assertRound2(-0.010, -0.015, 2);
    assertRound2(-0.020, -0.017, 2);
    assertRound2(-0.020, -0.022, 2);
    assertRound2(-0.020, -0.025, 2);
    assertRound2(-0.030, -0.027, 2);
    assertRound2(0.010, 0.010, 2);
    assertRound2(0.020, 0.020, 2);
    assertRound2(-0.030, -0.030, 2);

    assertRound2(100, 120, -2);
    assertRound2(200, 150, -2);
    assertRound2(200, 170, -2);
    assertRound2(200, 220, -2);
    assertRound2(300, 250, -2);
    assertRound2(300, 270, -2);
    assertRound2(-100, -120, -2);
    assertRound2(-100, -150, -2);
    assertRound2(-200, -170, -2);
    assertRound2(-200, -220, -2);
    assertRound2(-200, -250, -2);
    assertRound2(-300, -270, -2);
    assertRound2(100, 100, -2);
    assertRound2(200, 200, -2);
    assertRound2(-300, -300, -2);
  }

  @Test
  void testRoundStd1() throws HopException {
    assertRoundStd1(1.0, 1.2);
    assertRoundStd1(2.0, 1.5);
    assertRoundStd1(2.0, 1.7);
    assertRoundStd1(2.0, 2.2);
    assertRoundStd1(3.0, 2.5);
    assertRoundStd1(3.0, 2.7);
    assertRoundStd1(-1.0, -1.2);
    assertRoundStd1(-2.0, -1.5);
    assertRoundStd1(-2.0, -1.7);
    assertRoundStd1(-2.0, -2.2);
    assertRoundStd1(-3.0, -2.5);
    assertRoundStd1(-3.0, -2.7);
    assertRoundStd1(1.0, 1.0);
    assertRoundStd1(2.0, 2.0);
    assertRoundStd1(-3.0, -3.0);
  }

  @Test
  void testRoundStd2() throws HopException {
    assertRoundStd2(1.0, 1.2, 0);
    assertRoundStd2(2.0, 1.5, 0);
    assertRoundStd2(2.0, 1.7, 0);
    assertRoundStd2(2.0, 2.2, 0);
    assertRoundStd2(3.0, 2.5, 0);
    assertRoundStd2(3.0, 2.7, 0);
    assertRoundStd2(-1.0, -1.2, 0);
    assertRoundStd2(-2.0, -1.5, 0);
    assertRoundStd2(-2.0, -1.7, 0);
    assertRoundStd2(-2.0, -2.2, 0);
    assertRoundStd2(-3.0, -2.5, 0);
    assertRoundStd2(-3.0, -2.7, 0);
  }

  @Test
  void testRoundCustom1() throws HopException {
    assertRoundCustom1(2.0, 1.2, BigDecimal.ROUND_UP);
    assertRoundCustom1(1.0, 1.2, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(2.0, 1.2, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(1.0, 1.2, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(1.0, 1.2, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(1.0, 1.2, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(1.0, 1.2, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(1.0, 1.2, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(2.0, 1.5, BigDecimal.ROUND_UP);
    assertRoundCustom1(1.0, 1.5, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(2.0, 1.5, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(1.0, 1.5, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(2.0, 1.5, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(1.0, 1.5, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(2.0, 1.5, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(2.0, 1.5, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(2.0, 1.7, BigDecimal.ROUND_UP);
    assertRoundCustom1(1.0, 1.7, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(2.0, 1.7, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(1.0, 1.7, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(2.0, 1.7, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(2.0, 1.7, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(2.0, 1.7, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(2.0, 1.7, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(3.0, 2.2, BigDecimal.ROUND_UP);
    assertRoundCustom1(2.0, 2.2, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(3.0, 2.2, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(2.0, 2.2, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(2.0, 2.2, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(2.0, 2.2, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(2.0, 2.2, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(2.0, 2.2, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(3.0, 2.5, BigDecimal.ROUND_UP);
    assertRoundCustom1(2.0, 2.5, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(3.0, 2.5, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(2.0, 2.5, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(3.0, 2.5, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(2.0, 2.5, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(2.0, 2.5, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(3.0, 2.5, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(3.0, 2.7, BigDecimal.ROUND_UP);
    assertRoundCustom1(2.0, 2.7, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(3.0, 2.7, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(2.0, 2.7, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(3.0, 2.7, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(3.0, 2.7, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(3.0, 2.7, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(3.0, 2.7, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(-2.0, -1.2, BigDecimal.ROUND_UP);
    assertRoundCustom1(-1.0, -1.2, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(-1.0, -1.2, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(-2.0, -1.2, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(-1.0, -1.2, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(-1.0, -1.2, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(-1.0, -1.2, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(-1.0, -1.2, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(-2.0, -1.5, BigDecimal.ROUND_UP);
    assertRoundCustom1(-1.0, -1.5, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(-1.0, -1.5, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(-2.0, -1.5, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(-2.0, -1.5, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(-1.0, -1.5, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(-2.0, -1.5, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(-1.0, -1.5, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(-2.0, -1.7, BigDecimal.ROUND_UP);
    assertRoundCustom1(-1.0, -1.7, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(-1.0, -1.7, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(-2.0, -1.7, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(-2.0, -1.7, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(-2.0, -1.7, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(-2.0, -1.7, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(-2.0, -1.7, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(-3.0, -2.2, BigDecimal.ROUND_UP);
    assertRoundCustom1(-2.0, -2.2, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(-2.0, -2.2, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(-3.0, -2.2, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(-2.0, -2.2, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(-2.0, -2.2, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(-2.0, -2.2, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(-2.0, -2.2, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(-3.0, -2.5, BigDecimal.ROUND_UP);
    assertRoundCustom1(-2.0, -2.5, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(-2.0, -2.5, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(-3.0, -2.5, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(-3.0, -2.5, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(-2.0, -2.5, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(-2.0, -2.5, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(-2.0, -2.5, Const.ROUND_HALF_CEILING);

    assertRoundCustom1(-3.0, -2.7, BigDecimal.ROUND_UP);
    assertRoundCustom1(-2.0, -2.7, BigDecimal.ROUND_DOWN);
    assertRoundCustom1(-2.0, -2.7, BigDecimal.ROUND_CEILING);
    assertRoundCustom1(-3.0, -2.7, BigDecimal.ROUND_FLOOR);
    assertRoundCustom1(-3.0, -2.7, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom1(-3.0, -2.7, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom1(-3.0, -2.7, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom1(-3.0, -2.7, Const.ROUND_HALF_CEILING);
  }

  @Test
  void testRoundCustom2() throws HopException {
    assertRoundCustom2(2.0, 1.2, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(1.0, 1.2, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(2.0, 1.2, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(1.0, 1.2, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(1.0, 1.2, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(1.0, 1.2, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(1.0, 1.2, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(1.0, 1.2, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(2.0, 1.5, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(1.0, 1.5, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(2.0, 1.5, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(1.0, 1.5, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(2.0, 1.5, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(1.0, 1.5, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(2.0, 1.5, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(2.0, 1.5, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(2.0, 1.7, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(1.0, 1.7, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(2.0, 1.7, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(1.0, 1.7, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(2.0, 1.7, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(2.0, 1.7, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(2.0, 1.7, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(2.0, 1.7, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(3.0, 2.2, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(2.0, 2.2, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(3.0, 2.2, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(2.0, 2.2, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(2.0, 2.2, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(2.0, 2.2, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(2.0, 2.2, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(2.0, 2.2, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(3.0, 2.5, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(2.0, 2.5, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(3.0, 2.5, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(2.0, 2.5, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(3.0, 2.5, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(2.0, 2.5, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(2.0, 2.5, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(3.0, 2.5, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(3.0, 2.7, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(2.0, 2.7, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(3.0, 2.7, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(2.0, 2.7, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(3.0, 2.7, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(3.0, 2.7, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(3.0, 2.7, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(3.0, 2.7, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(-2.0, -1.2, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(-1.0, -1.2, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(-1.0, -1.2, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(-2.0, -1.2, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(-1.0, -1.2, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(-1.0, -1.2, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(-1.0, -1.2, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(-1.0, -1.2, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(-2.0, -1.5, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(-1.0, -1.5, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(-1.0, -1.5, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(-2.0, -1.5, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(-2.0, -1.5, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(-1.0, -1.5, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(-2.0, -1.5, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(-1.0, -1.5, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(-2.0, -1.7, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(-1.0, -1.7, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(-1.0, -1.7, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(-2.0, -1.7, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(-2.0, -1.7, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(-2.0, -1.7, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(-2.0, -1.7, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(-2.0, -1.7, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(-3.0, -2.2, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(-2.0, -2.2, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(-2.0, -2.2, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(-3.0, -2.2, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(-2.0, -2.2, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(-2.0, -2.2, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(-2.0, -2.2, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(-2.0, -2.2, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(-3.0, -2.5, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(-2.0, -2.5, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(-2.0, -2.5, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(-3.0, -2.5, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(-3.0, -2.5, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(-2.0, -2.5, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(-2.0, -2.5, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(-2.0, -2.5, 0, Const.ROUND_HALF_CEILING);

    assertRoundCustom2(-3.0, -2.7, 0, BigDecimal.ROUND_UP);
    assertRoundCustom2(-2.0, -2.7, 0, BigDecimal.ROUND_DOWN);
    assertRoundCustom2(-2.0, -2.7, 0, BigDecimal.ROUND_CEILING);
    assertRoundCustom2(-3.0, -2.7, 0, BigDecimal.ROUND_FLOOR);
    assertRoundCustom2(-3.0, -2.7, 0, BigDecimal.ROUND_HALF_UP);
    assertRoundCustom2(-3.0, -2.7, 0, BigDecimal.ROUND_HALF_DOWN);
    assertRoundCustom2(-3.0, -2.7, 0, BigDecimal.ROUND_HALF_EVEN);
    assertRoundCustom2(-3.0, -2.7, 0, Const.ROUND_HALF_CEILING);
  }

  public void assertRoundGeneral(
      final Object expectedResult,
      final CalculationType calcFunction,
      final Number value,
      final Long precision,
      final Long roundingMode,
      final int valueDataType,
      final int functionDataType) {

    final String msg =
        getHopTypeName(valueDataType) + "->" + getHopTypeName(functionDataType) + " ";

    final RowMeta inputRowMeta = new RowMeta();
    final List<Object> inputValues = new ArrayList<>(3);

    final String fieldValue = "Value";
    final IValueMeta valueMeta;
    switch (valueDataType) {
      case IValueMeta.TYPE_BIGNUMBER:
        valueMeta = new ValueMetaBigNumber(fieldValue);
        break;
      case IValueMeta.TYPE_NUMBER:
        valueMeta = new ValueMetaNumber(fieldValue);
        break;
      case IValueMeta.TYPE_INTEGER:
        valueMeta = new ValueMetaInteger(fieldValue);
        break;
      default:
        throw new IllegalArgumentException(
            msg
                + "Unexpected value dataType: "
                + value.getClass().getName()
                + ". Long, Double or BigDecimal expected.");
    }
    inputRowMeta.addValueMeta(valueMeta);
    inputValues.add(value);

    final String fieldPrecision;
    final ValueMetaInteger precisionMeta;
    if (precision == null) {
      fieldPrecision = null;
      precisionMeta = null;
    } else {
      fieldPrecision = "Precision";
      precisionMeta = new ValueMetaInteger(fieldPrecision);
      inputRowMeta.addValueMeta(precisionMeta);
      inputValues.add(precision);
    }

    final String fieldRoundingMode;
    final ValueMetaInteger roundingModeMeta;
    if (roundingMode == null) {
      fieldRoundingMode = null;
      roundingModeMeta = null;
    } else {
      fieldRoundingMode = "RoundingMode";
      roundingModeMeta = new ValueMetaInteger(fieldRoundingMode);
      inputRowMeta.addValueMeta(roundingModeMeta);
      inputValues.add(roundingMode);
    }

    IRowSet inputRowSet = smh.getMockInputRowSet(inputValues.toArray());
    inputRowSet.setRowMeta(inputRowMeta);
    final String fieldA =
        !inputRowMeta.isEmpty() ? inputRowMeta.getValueMetaList().get(0).getName() : null;
    final String fieldB =
        inputRowMeta.size() > 1 ? inputRowMeta.getValueMetaList().get(1).getName() : null;
    final String fieldC =
        inputRowMeta.size() > 2 ? inputRowMeta.getValueMetaList().get(2).getName() : null;

    final int resultDataType = functionDataType;

    final String fieldResult = "test";
    final int expectedResultRowSize = inputRowMeta.size() + 1;

    CalculatorMeta meta = new CalculatorMeta();
    meta.getFunctions()
        .add(
            new CalculatorMetaFunction(
                fieldResult,
                calcFunction,
                fieldA,
                fieldB,
                fieldC,
                ValueMetaFactory.getValueMetaName(resultDataType),
                2,
                0,
                "",
                "",
                "",
                "",
                false));

    CalculatorData data = new CalculatorData();

    Calculator calculator =
        new Calculator(smh.transformMeta, meta, data, 0, smh.pipelineMeta, smh.pipeline);
    calculator.addRowSetToInputRowSets(inputRowSet);
    calculator.setInputRowMeta(inputRowMeta);
    calculator.init();

    // Verify output
    try {
      calculator.addRowListener(
          new RowAdapter() {
            @Override
            public void rowWrittenEvent(IRowMeta rowMeta, Object[] row) {
              customAssertEquals(expectedResultRowSize, rowMeta.size());
              final int fieldResultIndex = rowMeta.size() - 1;
              customAssertEquals(fieldResult, rowMeta.getValueMeta(fieldResultIndex).getName());
              customAssertEquals(expectedResult, row[fieldResultIndex]);
            }
          });
      calculator.processRow();
    } catch (HopException ke) {
      ke.printStackTrace();
      fail(msg + ke.getMessage());
    }
  }

  /**
   * Asserts different data types according to specified expectedResult and value.<br>
   * Double - TYPE_NUMBER, TYPE_BIGNUMBER<br>
   * Integer - TYPE_NUMBER, TYPE_BIGNUMBER, TYPE_INTEGER
   *
   * @param expectedResult Double and Integer values allowed
   * @param calcFunction
   * @param value
   * @param precision
   * @param roundingMode
   * @throws HopException
   */
  public void assertRoundEveryDataType(
      final Number expectedResult,
      final CalculationType calcFunction,
      final Number value,
      final Long precision,
      final Long roundingMode)
      throws HopException {
    {
      final double resultValue = expectedResult.doubleValue();
      assertRoundGeneral(
          resultValue,
          calcFunction,
          value.doubleValue(),
          precision,
          roundingMode,
          IValueMeta.TYPE_NUMBER,
          IValueMeta.TYPE_NUMBER);
      assertRoundGeneral(
          resultValue,
          calcFunction,
          new BigDecimal(String.valueOf(value.doubleValue())),
          precision,
          roundingMode,
          IValueMeta.TYPE_BIGNUMBER,
          IValueMeta.TYPE_NUMBER);
      if (isInt(value)) {
        assertRoundGeneral(
            resultValue,
            calcFunction,
            value.longValue(),
            precision,
            roundingMode,
            IValueMeta.TYPE_INTEGER,
            IValueMeta.TYPE_NUMBER);
      }
    }
    {
      final BigDecimal resultValue = BigDecimal.valueOf(expectedResult.doubleValue());
      assertRoundGeneral(
          resultValue,
          calcFunction,
          value.doubleValue(),
          precision,
          roundingMode,
          IValueMeta.TYPE_NUMBER,
          IValueMeta.TYPE_BIGNUMBER);
      assertRoundGeneral(
          resultValue,
          calcFunction,
          new BigDecimal(String.valueOf(value.doubleValue())),
          precision,
          roundingMode,
          IValueMeta.TYPE_BIGNUMBER,
          IValueMeta.TYPE_BIGNUMBER);
      if (isInt(value)) {
        assertRoundGeneral(
            resultValue,
            calcFunction,
            value.longValue(),
            precision,
            roundingMode,
            IValueMeta.TYPE_INTEGER,
            IValueMeta.TYPE_BIGNUMBER);
      }
    }
    if (isInt(expectedResult)) {
      final Long resultValue = expectedResult.longValue();
      assertRoundGeneral(
          resultValue,
          calcFunction,
          value.doubleValue(),
          precision,
          roundingMode,
          IValueMeta.TYPE_NUMBER,
          IValueMeta.TYPE_INTEGER);
      assertRoundGeneral(
          resultValue,
          calcFunction,
          new BigDecimal(String.valueOf(value.doubleValue())),
          precision,
          roundingMode,
          IValueMeta.TYPE_BIGNUMBER,
          IValueMeta.TYPE_INTEGER);
      if (isInt(value)) {
        assertRoundGeneral(
            resultValue,
            calcFunction,
            value.longValue(),
            precision,
            roundingMode,
            IValueMeta.TYPE_INTEGER,
            IValueMeta.TYPE_INTEGER);
      }
    }
  }

  public void assertRound1(final Number expectedResult, final Number value) throws HopException {
    assertRoundEveryDataType(expectedResult, CalculationType.ROUND_1, value, null, null);
  }

  public void assertRound2(final Number expectedResult, final Number value, final long precision)
      throws HopException {
    assertRoundEveryDataType(expectedResult, CalculationType.ROUND_2, value, precision, null);
  }

  public void assertRoundStd1(final Number expectedResult, final Number value) throws HopException {
    assertRoundEveryDataType(expectedResult, CalculationType.ROUND_STD_1, value, null, null);
  }

  public void assertRoundStd2(final Number expectedResult, final Number value, final long precision)
      throws HopException {
    assertRoundEveryDataType(expectedResult, CalculationType.ROUND_STD_2, value, precision, null);
  }

  public void assertRoundCustom1(
      final Number expectedResult, final Number value, final long roundingMode)
      throws HopException {
    assertRoundEveryDataType(
        expectedResult, CalculationType.ROUND_CUSTOM_1, value, null, roundingMode);
  }

  public void assertRoundCustom2(
      final Number expectedResult,
      final Number value,
      final long precision,
      final long roundingMode)
      throws HopException {
    assertRoundEveryDataType(
        expectedResult, CalculationType.ROUND_CUSTOM_2, value, precision, roundingMode);
  }

  /**
   * Check whether value represents a whole number
   *
   * @param value
   * @return
   */
  private static boolean isInt(Number value) {
    if (value instanceof Long
        || value instanceof Integer
        || value instanceof Short
        || value instanceof Byte
        || value instanceof BigInteger) {
      return true;
    }
    final BigDecimal bigDecimalValue;
    if (value instanceof Double || value instanceof Float) {
      bigDecimalValue = new BigDecimal(value.toString());
    } else if (value instanceof BigDecimal) {
      bigDecimalValue = (BigDecimal) value;
    } else {
      throw new IllegalArgumentException("Unexpected dataType: " + value.getClass().getName());
    }
    try {
      bigDecimalValue.longValueExact();
      return true;
    } catch (ArithmeticException e) {
      return false;
    }
  }

  private String getHopTypeName(int kettleNumberDataType) {
    final String kettleNumberDataTypeName;
    switch (kettleNumberDataType) {
      case IValueMeta.TYPE_BIGNUMBER:
        kettleNumberDataTypeName = "BigNumber(" + kettleNumberDataType + ")";
        break;
      case IValueMeta.TYPE_NUMBER:
        kettleNumberDataTypeName = "Number(" + kettleNumberDataType + ")";
        break;
      case IValueMeta.TYPE_INTEGER:
        kettleNumberDataTypeName = "Integer(" + kettleNumberDataType + ")";
        break;
      default:
        kettleNumberDataTypeName = "?(" + kettleNumberDataType + ")";
    }
    return kettleNumberDataTypeName;
  }

  public static void customAssertEquals(Object expected, Object actual) {
    if (expected instanceof BigDecimal && actual instanceof BigDecimal) {
      if (((BigDecimal) expected).compareTo((BigDecimal) actual) != 0) {
        assertEquals(expected, actual);
      }
    } else {
      assertEquals(expected, actual);
    }
  }

  @Test
  void calculatorReminder() throws Exception {
    assertCalculatorReminder(
        Double.valueOf("0.10000000000000053"),
        new Object[] {Long.valueOf("10"), Double.valueOf("3.3")},
        new int[] {IValueMeta.TYPE_INTEGER, IValueMeta.TYPE_NUMBER});
    assertCalculatorReminder(
        Double.valueOf("1.0"),
        new Object[] {Long.valueOf("10"), Double.valueOf("4.5")},
        new int[] {IValueMeta.TYPE_INTEGER, IValueMeta.TYPE_NUMBER});
    assertCalculatorReminder(
        Double.valueOf("4.0"),
        new Object[] {Double.valueOf("12.5"), Double.valueOf("4.25")},
        new int[] {IValueMeta.TYPE_NUMBER, IValueMeta.TYPE_NUMBER});
    assertCalculatorReminder(
        Double.valueOf("2.6000000000000005"),
        new Object[] {Double.valueOf("12.5"), Double.valueOf("3.3")},
        new int[] {IValueMeta.TYPE_NUMBER, IValueMeta.TYPE_NUMBER});
  }

  private void assertCalculatorReminder(
      final Object expectedResult, final Object[] values, final int[] types) throws Exception {
    RowMeta inputRowMeta = new RowMeta();
    for (int i = 0; i < types.length; i++) {
      switch (types[i]) {
        case IValueMeta.TYPE_BIGNUMBER:
          inputRowMeta.addValueMeta(new ValueMetaBigNumber("f" + i));
          break;
        case IValueMeta.TYPE_NUMBER:
          inputRowMeta.addValueMeta(new ValueMetaNumber("f" + i));
          break;
        case IValueMeta.TYPE_INTEGER:
          inputRowMeta.addValueMeta(new ValueMetaInteger("f" + i));
          break;
        default:
          throw new IllegalArgumentException(
              "Unexpected value dataType: " + types[i] + ". Long, Double or BigDecimal expected.");
      }
    }

    IRowSet inputRowSet = null;
    try {
      inputRowSet = smh.getMockInputRowSet(new Object[][] {{values[0], values[1]}});
    } catch (Exception pe) {
      pe.printStackTrace();
      fail();
    }
    inputRowSet.setRowMeta(inputRowMeta);

    CalculatorMeta meta = new CalculatorMeta();
    meta.getFunctions()
        .add(
            new CalculatorMetaFunction(
                "res",
                CalculationType.REMAINDER,
                "f0",
                "f1",
                null,
                "Number",
                0,
                0,
                "",
                "",
                "",
                "",
                false));

    CalculatorData data = new CalculatorData();

    Calculator calculator =
        new Calculator(smh.transformMeta, meta, data, 0, smh.pipelineMeta, smh.pipeline);
    calculator.addRowSetToInputRowSets(inputRowSet);
    calculator.setInputRowMeta(inputRowMeta);
    calculator.init();

    // Verify output
    try {
      calculator.addRowListener(
          new RowAdapter() {
            @Override
            public void rowWrittenEvent(IRowMeta rowMeta, Object[] row)
                throws HopTransformException {
              try {
                customAssertEquals(expectedResult, row[2]);
              } catch (Exception pe) {
                throw new HopTransformException(pe);
              }
            }
          });
      calculator.processRow();
    } catch (HopException ke) {
      ke.printStackTrace();
      fail();
    }
  }
}
