/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.druid.k8s.overlord.common.httpclient.okhttp;

import io.fabric8.kubernetes.client.okhttp.OkHttpClientFactory;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;
import org.apache.druid.java.util.common.concurrent.Execs;
import org.apache.druid.k8s.overlord.common.httpclient.DruidKubernetesHttpClientFactory;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class DruidKubernetesOkHttpHttpClientFactory extends OkHttpClientFactory implements DruidKubernetesHttpClientFactory
{
  public static final String TYPE_NAME = "okhttp";
  private final DruidKubernetesOkHttpHttpClientConfig config;

  public DruidKubernetesOkHttpHttpClientFactory(DruidKubernetesOkHttpHttpClientConfig config)
  {
    super();
    this.config = config;
  }

  @Override
  protected void additionalConfig(OkHttpClient.Builder builder)
  {
    if (config.isUseCustomDispatcherExecutor()) {
      ExecutorService executorService = new ThreadPoolExecutor(
          config.getCoreWorkerThreads(),
          config.getMaxWorkerThreads(),
          config.getWorkerThreadKeepAliveTime(),
          TimeUnit.SECONDS,
          new LinkedBlockingQueue<>(),
          Execs.makeThreadFactory("OkHttpHttpClient-%d")
      );
      Dispatcher dispatcher = new Dispatcher(executorService);
      dispatcher.setMaxRequests(config.getMaxWorkerThreads());
      dispatcher.setMaxRequestsPerHost(config.getMaxWorkerThreads());
      builder.dispatcher(dispatcher);
    }
  }
}
